/*************************************************************************
*
*  File Name   : MousTail.c
*
*  Description : This program enables the PM mouse pointer to have a
*                moving "tail".
*
*
*  NOTE:       : Source is based on the DIALOG sample (OS/2 2.0).
*
*
*  History:    :
*
*  7 Jun 93    Ari Erev  Created.
*
************************************************************************/

#define INCL_WINSYS                     /* Selectively include          */
#define INCL_WINFRAMEMGR                /* relevant parts of            */
#define INCL_WINMENUS                   /* the PM header file           */
#define INCL_WINDIALOGS
#define INCL_WINPOINTERS
#define INCL_WINSTDSLIDER
#define INCL_WINTIMER
#define INCL_GPI

#include <os2.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "moustail.h"                   /* Resource symbolic identifiers*/

#define LENGTH_STRING  33               /* Length of string             */
#define RETURN_ERROR    1               /* return value for exit        */
#define BEEP_WARN_FREQ      60          /* frequency of warning beep    */
#define BEEP_WARN_DUR      100          /* duration of warning beep     */

#define ID_TIMER   1

/* Function prototypes                                                  */
int main(VOID);

MRESULT EXPENTRY MouseTailWndProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 );
MRESULT EXPENTRY SetSpeedDlgProc( HWND hwndDlg, USHORT msg,
                                   MPARAM mp1, MPARAM mp2 );
MRESULT EXPENTRY AboutDlgProc( HWND hwndDlg, USHORT msg, MPARAM mp1, MPARAM mp2 );

BOOL    InitializeSlider(HWND hwnd);
VOID    DisplayMessage( PCH str );
VOID    SetSysMenu(HWND hwnd);
VOID    SetSystemMenu(HWND hwnd);
VOID    Set_MouseTail_Pointer(HWND hwnd);
VOID    Set_MouseTail_Speed  (HWND hwnd);
ULONG   QuerySliderPos( HWND hwndDlg);
VOID    Load_Strings(VOID);
VOID    SetMoveMenuItem( PCHAR pchText );

BOOL    TailMoving = TRUE;
                                        /* Define parameters by type    */

HAB   hab;                              /* Anchor block handle          */
HMQ   hmq;                              /* Message queue handle         */
HWND  hwndClient;                       /* Client window handle         */
ULONG flCreate = FCF_STANDARD &         /* Frame control flag           */
                  ~(FCF_MENU);

BOOL  bComplete = FALSE;                /* Switch for first time through*/
HWND  hwndFrame;                        /* Frame window handle          */
CHAR  szLocation[LENGTH_STRING] = "";   /* To hold text entered by user */
CHAR  szResponse[LENGTH_STRING] = "";   /* To hold "You live in"        */


ULONG MouseTail_Speed = 8L;             /* Devisor to get milisecond values */

CHAR  szStopTail[LENGTH_STRING];
CHAR  szMoveTail[LENGTH_STRING];
CHAR  szSetTailSpeed[LENGTH_STRING];
CHAR  szAppName[LENGTH_STRING];
CHAR  szAbout[LENGTH_STRING];

/****************************************************************
 *  Name:   main()
 *
 *  Description: Entry point of program.
 *
 *  Concepts: Obtains anchor block handle and creates message
 *            queue.  Calls the initialization routine.
 *            Creates the main frame window which creates the
 *            main client window.  Polls the message queue
 *            via Get/Dispatch Msg loop.  Upon exiting the
 *            loop, exits.
 *
\****************************************************************/
int main(VOID)
{
  QMSG  qmsg;
  BOOL success;

  hab = WinInitialize( 0 );
  if(!hab)
  {
    DosBeep(BEEP_WARN_FREQ, BEEP_WARN_DUR);
    exit(RETURN_ERROR);
  }

  hmq = WinCreateMsgQueue( hab, 0 );
  if(!hmq)
  {
    DosBeep(BEEP_WARN_FREQ, BEEP_WARN_DUR);
    WinTerminate(hab);
    exit(RETURN_ERROR);
  }


  success = WinRegisterClass(           /* Register window class        */
       hab,                             /* Anchor block handle          */
       "MousTail",                      /* Window class name            */
       MouseTailWndProc,                /* Address of window procedure  */
       CS_SIZEREDRAW,                   /* Class style                  */
       0                                /* No extra window words        */
       );
  if (!success)
  {
    DisplayMessage("Could Not Register Class");
    exit(RETURN_ERROR);
  }

  hwndFrame = WinCreateStdWindow(
        HWND_DESKTOP,                   /* Desktop window is parent     */
   //   WS_VISIBLE,                     /* Frame style                  */
        0,                              /* Frame style                  */
        &flCreate,                      /* Frame control flag           */
        "MousTail",                     /* Window class name            */
        "",                             /* Window title                 */
        0L,                             /* Client style of VISIBLE      */
        (HMODULE)NULL,                  /* Resource is in .EXE file     */
        ID_MAINWND,                     /* Frame window identifier      */
        &hwndClient);                   /* Client window handle         */

  if (!hwndFrame)
  {
    DisplayMessage("Could Not Create Main Window");
    exit(RETURN_ERROR);
  }

  Load_Strings();

  WinSetWindowText(hwndFrame, szAppName);

  WinSetWindowPos ( hwndFrame,
                    HWND_TOP,
                    0,0,0,0,
                    SWP_MINIMIZE | SWP_SHOW );

  SetSystemMenu(hwndFrame);


/*
 * Get and dispatch messages from the application message queue
 * until WinGetMsg returns FALSE, indicating a WM_QUIT message.
 */
  while( WinGetMsg( hab, &qmsg, (HWND)NULL, 0, 0 ) )
    WinDispatchMsg( hab, &qmsg );

  WinDestroyWindow( hwndFrame );        /* Tidy up...                   */
  WinDestroyMsgQueue( hmq );            /* and                          */
  WinTerminate( hab );                  /* terminate the application    */
  return(0);
}
/***********************  End of main procedure  ************************/

/****************************************************************
 *  Name:   MouseTailWndProc
 *
 *  Description : Window procedure for the main clent window.
 *
 *  Concepts : Processes the messages sent to the main client
 *             window.
 *
 ****************************************************************/
MRESULT EXPENTRY MouseTailWndProc ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  switch( msg )
  {

    case WM_CREATE:
         WinPostMsg(hwnd, WMU_SET_SPEED, NULL, NULL);
         break;

    case WMU_SET_SPEED:
         Set_MouseTail_Speed(hwnd);
         break;


    case WM_TIMER:
         if (TailMoving)
             Set_MouseTail_Pointer(hwnd);
         break;



    case WM_PAINT:
      /*
       * Window contents are drawn here. First time through, bComplete
       * is FALSE, so window is simply filled with SYSCLR_WINDOW.
       * On subsequent passes, if bComplete has been set to TRUE in the
       * dialog procedure, GpiCharStringAt draws the text.
       */
      {
       HPS    hps;                      /* Presentation space handle    */
       POINTL pt;                       /* String screen coordinates    */
       RECTL  rc;                       /* Window rectangle             */

        hps = WinBeginPaint( hwnd, (HPS)NULL, &rc );
        WinFillRect( hps, &rc, SYSCLR_WINDOW );
        WinEndPaint( hps );               /* Drawing is complete          */
      }
      break;

    case WM_COMMAND:
      /*
       * Gets commands from system menu.
       */
      switch ( SHORT1FROMMP( mp1 ) )
      {
        case ID_EXITPROGRAM:
             WinPostMsg( hwnd, WM_CLOSE, NULL, NULL);
             break;

        case IDM_SET_SPEED:

             /* Display the "Set Speed" Dialog */

             WinDlgBox( HWND_DESKTOP,      /* Parent                    */
                hwndFrame,                 /* Owner                     */
                (PFNWP)SetSpeedDlgProc,    /* Address of dialog proc    */
                NULLHANDLE,                /* Module handle             */
                IDD_SETSPEED_DLG,          /* Id of dialog in resource  */
                (PVOID)0L );               /* Initialisation data       */

             WinPostMsg(hwnd, WMU_SET_SPEED, NULL, NULL);
             break;

        case IDM_START_STOP_TAIL:  /* Toggle Move/NoMove state. */
             {
             PCHAR pch;

             if (TailMoving) {
                 TailMoving = FALSE;
                 pch        = szMoveTail;
                 WinStopTimer (hab, hwnd, ID_TIMER);
              } else {
                 TailMoving = TRUE;
                 pch        = szStopTail;
                 WinPostMsg(hwnd, WMU_SET_SPEED, NULL, NULL);
              }

              SetMoveMenuItem(pch);

              }
              break;

        case IDM_ABOUT:
             WinDlgBox( HWND_DESKTOP,      /* Parent                    */
                hwndFrame,                 /* Owner                     */
                (PFNWP)AboutDlgProc,       /* Address of dialog proc    */
                NULLHANDLE,                /* Module handle             */
                IDD_ABOUT_DLG,             /* Id of dialog in resource  */
                (PVOID)0L );               /* Initialisation data       */

             break;


        /* This command should be ignored */
        /* In effect, they do not get here at all... */
        case SC_SIZE:
        case SC_MAXIMIZE:
        case SC_MINIMIZE:
        case SC_RESTORE:
             break;

      }
      break;

    case WM_CLOSE:
      WinPostMsg( hwnd, WM_QUIT, NULL, NULL);     /* Cause termination    */
      break;

    default:
      return WinDefWindowProc( hwnd, msg, mp1, mp2 );

  } /* end switch */

  return (MRESULT) FALSE;

} /* MouseTailWndProc() */


/****************************************************************
 *  Name:   DisplayMessage
 *
 *  Description : Displays error messages.
 *
 ****************************************************************/
VOID DisplayMessage(PCH str)
{
  /*
   * This routine calls WinMessageBox to dsplay error messages to the
   * user.
   */

  WinMessageBox(HWND_DESKTOP, HWND_DESKTOP, (PCH) str,
     (PCH) "Error Messages For The Dialog Sample", ID_MESSAGEBOX,
     MB_OK | MB_APPLMODAL | MB_MOVEABLE );

}  /* DispalyMessage() */



/****************************************************************
 *  Name:   SetSpeedDlgProc
 *
 *  Description : Dialog procedure for the "Set Speed" dialog.
 *
 ****************************************************************/
MRESULT EXPENTRY SetSpeedDlgProc( HWND hwndDlg, USHORT msg,
                                   MPARAM mp1, MPARAM mp2 )
{
  switch (msg)
  {

    case WM_INITDLG:
      InitializeSlider(hwndDlg);
      return FALSE;
      break;


    case WM_COMMAND:
      switch( SHORT1FROMMP( mp1 ) )
      {
        case DID_OK:    /* Enter key or pushbutton pressed/ selected */

          MouseTail_Speed = QuerySliderPos(hwndDlg);

        case DID_CANCEL: /* Escape key or CANCEL pushbutton pressed/selected */
          break;
        default:
          return WinDefDlgProc( hwndDlg, msg, mp1, mp2 );
      }
      WinDismissDlg( hwndDlg, TRUE ); /* Finished with dialog box    */
      break;

    default:  /* Pass all other messages to the default dialog proc  */
      return WinDefDlgProc( hwndDlg, msg, mp1, mp2 );
  }
  return FALSE;

} /* SetSpeedDlgPro() */

/****************************************************************
 *  Name:   AboutDlgProc
 *
 *  Description : Dialog procedure for the "About" dialog.
 *
 ****************************************************************/
MRESULT EXPENTRY AboutDlgProc( HWND hwndDlg, USHORT msg,
                               MPARAM mp1, MPARAM mp2 )
{
   return WinDefDlgProc( hwndDlg, msg, mp1, mp2 );

} /* AboutDlgPro() */



/* Table of hptrs that contain all the pointer handles of the "moving tail" */
HWND MyPtrs[7] = { 0, 0, 0, 0, 0, 0, 0, };

/****************************************************************
 *  Name:   Set_MouseTail_Pointer
 *
 *  Description : Sets a new mouse pointer.
 *
 *                This is actually the "heart" of the program.
 *
 *                Checks to see if the current pointer is one of our
 *                own pointer or the SPTR_ARROW pointer.
 *                if yes - relaces the pointer with one of our own pointers.
 *                else   - leave the pointer as-is.
 *
 *
 *
 ****************************************************************/
VOID     Set_MouseTail_Pointer(HWND hwnd)
{

static HWND NextPointerID=0;  /* Contains the "next" pointer handle */
static SHORT Direction = -1;  /* Instructs the routine what the NextPointer */
                              /* should be.                                 */

static HWND hptrArrow = 0;    /* Saves the SPTR_ARROW pointer handle.       */
       HWND hptrCurrent;      /* The current pointer handle.                */

int i;


//   if (NextPointerID > ID_MAX_POINTER) {
//
//       NextPointerID = ID_MAXMIN_POINTER;
//
//   if (NextPointerID == 0L)
//       NextPointerID = ID_MIN_POINTER;

     if (NextPointerID == 0) {
        /* Initialize */
        NextPointerID = ID_CENTER_POINTER;

        for (i=0; i < (sizeof(MyPtrs)/sizeof(HWND)) ; i++) {
             MyPtrs[i] = WinLoadPointer (HWND_DESKTOP,
                                         (ULONG)0,
                                         ID_MIN_POINTER + i );
        }

        /* Save system pointer */
        hptrArrow = WinQuerySysPointer(HWND_DESKTOP, SPTR_ARROW, FALSE);

     }


     /*--------------------------------------------------------------------*/
     /*  Get the current mouse pointer.                                    */
     /*--------------------------------------------------------------------*/
     hptrCurrent = WinQueryPointer(HWND_DESKTOP);


     /*--------------------------------------------------------------------*/
     /* If current pointer is the system-arrow, or one of our own pointers */
     /* then we replace it with the NextPointer.                           */
     /*--------------------------------------------------------------------*/
     if (hptrCurrent == hptrArrow ||

         hptrCurrent == MyPtrs[0] ||
         hptrCurrent == MyPtrs[1] ||
         hptrCurrent == MyPtrs[2] ||
         hptrCurrent == MyPtrs[3] ||
         hptrCurrent == MyPtrs[3] ||
         hptrCurrent == MyPtrs[4] ||
         hptrCurrent == MyPtrs[5] ||
         hptrCurrent == MyPtrs[6]    ) {

        WinSetPointer (HWND_DESKTOP,
                       MyPtrs [NextPointerID - ID_MIN_POINTER] );

        /*----------------------------------------------------------*/
        /* Set the handle of the NextPointer (for the next time we  */
        /* are called)                                              */
        /*----------------------------------------------------------*/
        if (NextPointerID == ID_MAX_POINTER) {
            Direction = -1;
        }

        if (NextPointerID == ID_MIN_POINTER) {
            Direction = 1;
        }

        NextPointerID = NextPointerID + Direction;

     } else { /* else, the mouse pointer should not be replaced */

        NextPointerID = ID_CENTER_POINTER; /* Re-initialize tail to be on center */

     } /* endif */

}


/****************************************************************
 *  Name:   Set_MouseTail_Speed
 *
 *  Description : Sets a the speed of the moving tail.
 *
 ****************************************************************/
VOID Set_MouseTail_Speed(HWND hwnd)
{
         WinStartTimer (hab, hwnd, ID_TIMER, (ULONG) (1000 / MouseTail_Speed) );
}


/*********************************************************************
 *  Name : InitializeSlider
 *
 *  Description :  Set the Sliders Tick size and Scale Text.
 *
 *******************************************************************/
BOOL InitializeSlider(HWND hwnd)
{
   USHORT usIndex;
   CHAR   acBuffer[4];


   /* Let tick go from 1 to 10 (including 10) */
   for (usIndex = 0;usIndex < 10 ;usIndex ++ )
   {
      _itoa(usIndex+1,acBuffer,10);
      if ( !WinSendDlgItemMsg(hwnd, ID_SET_SPEED,  SLM_SETTICKSIZE,
               MPFROM2SHORT(usIndex, 5), NULL))
        return FALSE;

      if ( !WinSendDlgItemMsg(hwnd, ID_SET_SPEED, SLM_SETSCALETEXT,
                 MPFROMSHORT(usIndex), MPFROMP(acBuffer)))
        return FALSE;
   }

   /*-------------------------------------------*/
   /* Set the arm to the current speed setting  */
   /*-------------------------------------------*/
   WinSendDlgItemMsg(hwnd, ID_SET_SPEED, SLM_SETSLIDERINFO,
                     MPFROM2SHORT(SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE),
                     MPFROMLONG(MouseTail_Speed-1) );

   return TRUE;

}  /* End of InitializeSlider      */



/* Gets the position of the slider (user's selection of speed) */
/*********************************************************************
 *  Name : QuerySliderPos
 *
 *  Description :  Gets the current position (in range values) of
 *                 the slider arm.
 *
 *******************************************************************/
ULONG QuerySliderPos(HWND hwndDlg)
{
ULONG ulValue;
   CHAR   acBuffer[4];

      ulValue = (ULONG) WinSendDlgItemMsg(hwndDlg,
                                          ID_SET_SPEED,
                                          SLM_QUERYSLIDERINFO,
                      MPFROM2SHORT(SMA_SLIDERARMPOSITION,SMA_INCREMENTVALUE),
                      NULL);



      acBuffer[0] = acBuffer[1] = acBuffer[2] = acBuffer[3] = 0x00;

      ulValue += 1;   /* because the slider is 0-index based   */
                      /* We should not get 0 as a return value */
                      /* because it is used as a divisor.      */

//      _ltoa(ulValue,acBuffer,10);
//      DisplayMessage(acBuffer);

      return ulValue;
}

/*********************************************************************
 *                                                                   *
 * Name:          SetSystemMenu                                      *
 *                                                                   *
 * Edit the system-menu items. adds right-justified Hebrew           *
 * translations to the system-menu items.                            *
 *                                                                   *
 *********************************************************************/
VOID  SetSystemMenu( HWND hwnd )
{
  HWND     hwndSysMenu;                 /* sys menu window    handle */
  HWND     hwndSubMenu;                 /* sys menu pull-down handle */
  MENUITEM miTemp;                      /* menu item template        */
  SHORT    sItemId;                     /* system menu item ID       */
  SHORT    sItemTextLength;             /* system menu item ID       */
  SHORT    sItemIndex;                  /* system menu item index    */

  MENUITEM miAboutMenuItem;

  ULONG    ulItemId;


  /*******************************************************************/
  /* Get the handle of the system menu pull-down.                    */
  /*******************************************************************/
  hwndSysMenu = WinWindowFromID( hwnd, FID_SYSMENU );

  WinSendMsg( hwndSysMenu,
              MM_QUERYITEM,
              MPFROM2SHORT( SC_SYSMENU, FALSE ),
              MPFROMP( (PSZ)&miTemp ));
  hwndSubMenu = miTemp.hwndSubMenu;

  /*******************************************************************/
  /* Add Hebrew text to the standard system menu entries.            */
  /*******************************************************************/
  for ( sItemIndex = 0, sItemId = 0; sItemId != MIT_ERROR; sItemIndex++)
  {
    sItemId = (SHORT)WinSendMsg( hwndSubMenu,
                                 MM_ITEMIDFROMPOSITION,
                                 MPFROMSHORT( sItemIndex ),
                                (MPARAM)NULL );

    ulItemId = (ULONG) (USHORT) sItemId;


    switch( ulItemId )
       {

       case SC_RESTORE:
       case SC_MAXIMIZE:
       case SC_MINIMIZE:
       case SC_SIZE:
          WinSendMsg(hwndSubMenu, MM_DELETEITEM,
                     MPFROM2SHORT(sItemId,FALSE),NULL);
          break;

       case SC_MOVE:
       case SC_HIDE:
       case SC_CLOSE:
       case SC_TASKMANAGER:
       case (-1):
       default:
          break;
       }  /* end switch */

  }

  miTemp.iPosition   = MIT_END;
  miTemp.afStyle     = MIS_SEPARATOR;
  miTemp.afAttribute = 0;
  miTemp.id          = -1;
  miTemp.hwndSubMenu = NULLHANDLE;
  miTemp.hItem       = 0;

  WinSendMsg(hwndSubMenu, MM_INSERTITEM,
             MPFROMP(&miTemp), NULL);

  miTemp.iPosition   = MIT_END;
  miTemp.afStyle     = MIS_TEXT;
  miTemp.afAttribute = 0;
  miTemp.id          = IDM_SET_SPEED;
  miTemp.hwndSubMenu = NULLHANDLE;
  miTemp.hItem       = 0;

  WinSendMsg(hwndSubMenu, MM_INSERTITEM,
             MPFROMP(&miTemp), MPFROMP(szSetTailSpeed) );

  miTemp.iPosition   = MIT_END;
  miTemp.afStyle     = MIS_TEXT;
  miTemp.afAttribute = 0;
  miTemp.id          = IDM_START_STOP_TAIL;
  miTemp.hwndSubMenu = NULLHANDLE;
  miTemp.hItem       = 0;

  WinSendMsg(hwndSubMenu, MM_INSERTITEM,
             MPFROMP(&miTemp), MPFROMP(szStopTail) );

  miTemp.iPosition   = MIT_END;
  miTemp.afStyle     = MIS_SEPARATOR;
  miTemp.afAttribute = 0;
  miTemp.id          = -1;
  miTemp.hwndSubMenu = NULLHANDLE;
  miTemp.hItem       = 0;

  WinSendMsg(hwndSubMenu, MM_INSERTITEM,
             MPFROMP(&miTemp), NULL);


  miTemp.iPosition   = MIT_END;
  miTemp.afStyle     = MIS_TEXT;
  miTemp.afAttribute = 0;
  miTemp.id          = IDM_ABOUT;
  miTemp.hwndSubMenu = NULLHANDLE;
  miTemp.hItem       = 0;

  WinSendMsg(hwndSubMenu, MM_INSERTITEM,
             MPFROMP(&miTemp), MPFROMP(szAbout) );


} /* SetSystemMenu() */

/*********************************************************************
 *                                                                   *
 * Name:          SetMoveMenuItem                                    *
 *                                                                   *
 * Sets the text of the Move/Stop tail menu item                     *
 *                                                                   *
 *********************************************************************/
VOID  SetMoveMenuItem( PCHAR pchText )
{
  HWND     hwndSysMenu;                 /* sys menu window    handle */
  HWND     hwndSubMenu;                 /* sys menu pull-down handle */
  MENUITEM miTemp;


  /*******************************************************************/
  /* Get the handle of the system menu pull-down.                    */
  /*******************************************************************/
  hwndSysMenu = WinWindowFromID( hwndFrame, FID_SYSMENU );

  WinSendMsg( hwndSysMenu,
              MM_QUERYITEM,
              MPFROM2SHORT( SC_SYSMENU, FALSE ),
              MPFROMP( (PSZ)&miTemp ));
  hwndSubMenu = miTemp.hwndSubMenu;

  /* Set the text */
  WinSendMsg (hwndSubMenu,
              MM_SETITEMTEXT,
              MPFROMSHORT(IDM_START_STOP_TAIL),
              MPFROMP(pchText) );


} /* SetMoveMenuItem() */

/******************************************************
* LoadStrings - loads all strings from resource file.
*******************************************************/
VOID Load_Strings()
{
 WinLoadString(hab,(HMODULE)NULL, IDMSG_APPNAME,  LENGTH_STRING, szAppName);
 WinLoadString(hab,(HMODULE)NULL, IDMSG_SETSPEED, LENGTH_STRING, szSetTailSpeed);
 WinLoadString(hab,(HMODULE)NULL, IDMSG_STOPTAIL, LENGTH_STRING, szStopTail);
 WinLoadString(hab,(HMODULE)NULL, IDMSG_MOVETAIL, LENGTH_STRING, szMoveTail);
 WinLoadString(hab,(HMODULE)NULL, IDMSG_ABOUT   , LENGTH_STRING, szAbout);
} /* LoadStrings() */


